//
//  CRCIHistogram.h
//  CRGraphics
//
//  Created by Yaroslav Glushchenko on 9/9/09.
//  Copyright 2009 Corner-A. All rights reserved.
//

//////////////////////////////////////////////////////////////////////
// Pixel structs and some constants
//////////////////////////////////////////////////////////////////////
typedef float HMFloat;	// 32-bit float type since NSBitmapImageRep doesn't support 64-bit version

struct Pixel
{
	HMFloat r, g, b, l;
	
	Pixel()
	{}
	
	Pixel(HMFloat _r, HMFloat _g, HMFloat _b, HMFloat _l)
		:r(_r), g(_g), b(_b), l(_l)
	{}
	
	inline Pixel& operator += (const Pixel& p)
	{
		r += p.r;
		g += p.g;
		b += p.b;
		l += p.l;
		return *this;
	}
	
	inline Pixel& operator *= (double v)
	{
		r *= v;
		g *= v;
		b *= v;
		l *= v;
		return *this;
	}
	
	inline Pixel& operator /= (double v)
	{
		return this->operator *= (1.0 / v);
	}
};

#define CRHistogramBucketsCount	256
#define CRHistogramQuality		96		// histogram source size (rescales input image before computance)
#define CRHistogramCritical		5		// values 'this' times bigger than average are ignored

//////////////////////////////////////////////////////////////////////
// Histogram
//////////////////////////////////////////////////////////////////////
typedef enum {
	CRHistogramRed		= 0x01,
	CRHistogramGreen	= 0x02,
	CRHistogramBlue		= 0x04,
	CRHistogramGray		= 0x08,
	
	CRHistogramRGB		= CRHistogramRed | CRHistogramGreen | CRHistogramBlue,
	CRHistogramAll		= CRHistogramRGB | CRHistogramGray,
} CRHistogramRepresentationType;

@interface CRCIHistogram : NSObject
{
	int					imageWidth;
	int					imageHeight;
	
	NSArray*			dependencyProps;
	CIImage*			inputImage;
	CIImage*			scaledInput;
	
	BOOL				isDirty;
	Pixel				histogram[CRHistogramBucketsCount];
	CIVector*			minimum;
	CIVector*			maximum;
	CIVector*			average;
	
	NSImage*			histogramImage;
	NSInteger			histogramImageBucketStep;
	NSInteger			histogramImageType;
	NSArray*			histogramChannelColors;
	NSColor*			histogramBackgroundColor;
	
	NSBitmapImageRep*	buffer1px;		// cached buffer to render 1-pixel data (min, max, average filters output)
	NSBitmapImageRep*	bufferHgm;		// cached buffer for the histogram render
}

@property (nonatomic, strong)	CIImage*	inputImage;

@property (nonatomic, assign)	int			imageWidth;
@property (nonatomic, assign)	int			imageHeight;

@property (nonatomic, readonly)	CIVector*	minimum;
@property (nonatomic, readonly)	CIVector*	maximum;
@property (nonatomic, readonly)	CIVector*	average;
@property (nonatomic, readonly) Pixel*		histogram;

@property (nonatomic, strong)	NSImage*	histogramImage;
@property (nonatomic, assign)	NSInteger	histogramImageBucketStep;
@property (nonatomic, assign)	NSInteger	histogramImageType;
@property (nonatomic, copy)		NSArray*	histogramChannelColors;
@property (nonatomic, copy)		NSColor*	histogramBackgroundColor;

@end